﻿using System;
using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEngine;

//MoneyFlyoutController manages the player money flyout
public class MoneyFlyoutController : MonoBehaviour
{
    private enum CurrentAnimation
    {
        Shrinking,
        Growing,
        Normal
    }

    public float AnimationTime = 2000.0f;   //How long the tick animation should take
    public Animator Animator;
    public TMP_Text MoneyText;

    private float _CurrentMoney;    //How much money we are showing
    private int _TargetMoney;   //What our anim to tick to is
    private float _TickAmount;  //How much to tick by each millisecond
    private CurrentAnimation _CurrentAnimation;
    private float _AnimationTimeElapsed;

    private AudioSource _TickSFXSource;

    /// <summary>
    /// Updates the displayed money by animating to the new value
    /// </summary>
    /// <param name="money">The new money value to display</param>
    public void UpdateMoney(int money)
    {
        if (_CurrentAnimation == CurrentAnimation.Normal && _CurrentMoney != money)
        {
            //Set our target, calculate our tick amount and play the audio
            _TargetMoney = money;
            _TickAmount = (_TargetMoney - _CurrentMoney) / AnimationTime;
            _TickSFXSource = AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles["MoneyTickSFX"], true);

            //Play the correct animation
            if (_TargetMoney < _CurrentMoney)
            {
                _CurrentAnimation = CurrentAnimation.Shrinking;
                Animator.Play("NormalToShrink");
            }

            else
            {
                _CurrentAnimation = CurrentAnimation.Growing;
                Animator.Play("NormalToGrow");
            }
        }
    }

    private void Start()
    {
        _CurrentMoney = 0;
        GameManager.Instance.GameStateChanged += OnGameStateChanged;
    }

    /// <summary>
    /// Handler for the game state changing
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void OnGameStateChanged(object sender, GameManager.GameStateChangedEventArgs e)
    {
        if (_TickSFXSource != null)
        {
            //Make sure we handle pausing and unpausing the audio
            if (e.NewState != Constants.GameState.Playing)
            {
                if (_TickSFXSource.isPlaying)
                {
                    AudioManager.Instance.PauseSFX(_TickSFXSource);
                }
            }

            else
            {
                if (AudioManager.Instance.IsSFXPaused(_TickSFXSource))
                {
                    AudioManager.Instance.UnPauseSFX(_TickSFXSource);
                }
            }
        }
    }

    private void Update()
    {
        if(_CurrentAnimation != CurrentAnimation.Normal)
        {
            //We're animating, let's update the money value and time elapsed based on the state
            float deltaTimeMS = (Time.deltaTime * 1000.0f);

            _CurrentMoney += (_TickAmount * deltaTimeMS);
            MoneyText.text = "$" + Convert.ToInt32(_CurrentMoney).ToString();
            _AnimationTimeElapsed += deltaTimeMS;

            if(_AnimationTimeElapsed >= AnimationTime)
            {
                //All done, stop the audio, snap the value and reset
                AudioManager.Instance.StopFile(AudioManager.Instance.LoadedFiles["MoneyTickSFX"]);
                _CurrentMoney = _TargetMoney;
                MoneyText.text = "$" + Convert.ToInt32(_CurrentMoney).ToString();
                _AnimationTimeElapsed = 0.0f;

                //Finally, play the correct animation and reset
                if (_CurrentAnimation == CurrentAnimation.Shrinking)
                {
                    Animator.Play("ShrinkToNormal");
                }

                else
                {
                    Animator.Play("GrowToNormal");
                }

                _CurrentAnimation = CurrentAnimation.Normal;
            }
        }
    }
}
